<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Review;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Display a listing of products
     */
    public function index(Request $request)
    {
        $query = Product::with('category')->active();

        // Category filter
        if ($request->has('category') && $request->category) {
            $query->where('category_id', $request->category);
        }

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }

        // Organic filter
        if ($request->has('organic') && $request->organic) {
            $query->where('is_organic', true);
        }

        // Featured filter
        if ($request->has('featured') && $request->featured) {
            $query->where('is_featured', true);
        }

        // Sort
        $sort = $request->get('sort', 'order');
        $direction = $request->get('direction', 'asc');
        
        switch ($sort) {
            case 'price_asc':
                $query->orderBy('sale_price', 'asc')->orderBy('price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('sale_price', 'desc')->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', $direction);
                break;
            case 'newest':
                $query->orderBy('created_at', 'desc');
                break;
            default:
                $query->orderBy('order')->orderBy('created_at', 'desc');
        }

        $products = $query->paginate(12)->withQueryString();
        $categories = \App\Models\Category::where('is_active', true)->orderBy('order')->get();

        return view('product.index', compact('products', 'categories'));
    }

    /**
     * Display the specified product
     */
    public function show($slug)
    {
        $product = Product::with(['category', 'approvedReviews.user'])
            ->where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        // Related products (same category or featured)
        $relatedProducts = Product::where('id', '!=', $product->id)
            ->where('is_active', true)
            ->where(function($query) use ($product) {
                $query->where('category_id', $product->category_id)
                      ->orWhere('is_featured', true);
            })
            ->inRandomOrder()
            ->limit(4)
            ->get();

        // Get approved reviews
        $reviews = $product->approvedReviews()->with('user')->latest()->paginate(10);
        
        // Check if user can review (has purchased and is logged in)
        $canReview = false;
        $userReview = null;
        if (auth()->check()) {
            $canReview = $product->hasUserPurchased(auth()->id());
            $userReview = Review::where('product_id', $product->id)
                ->where('user_id', auth()->id())
                ->first();
        }

        return view('product.show', compact('product', 'relatedProducts', 'reviews', 'canReview', 'userReview'));
    }

    /**
     * Search products
     */
    public function search(Request $request)
    {
        $searchTerm = $request->get('q', '');
        
        $query = Product::with('category')->active();

        if ($searchTerm) {
            $query->where(function($q) use ($searchTerm) {
                $q->where('name', 'like', '%' . $searchTerm . '%')
                  ->orWhere('description', 'like', '%' . $searchTerm . '%')
                  ->orWhere('description_detail', 'like', '%' . $searchTerm . '%')
                  ->orWhere('brand', 'like', '%' . $searchTerm . '%');
            });
        }

        $products = $query->orderBy('order')->orderBy('created_at', 'desc')->paginate(12)->withQueryString();
        $categories = \App\Models\Category::where('is_active', true)->orderBy('order')->get();

        return view('product.search', compact('products', 'categories', 'searchTerm'));
    }
}
